from __future__ import absolute_import
try:
    import simplejson as json
except ImportError as e:
    import json

from .fields.base_field import BaseField
from .functions import jsonize_str
from .functions import pythonize_str
from .decorators import with_metaclass


TAG = "Serializable"

_field_name = "_jsoner_data"


class MetaSerializable(type):

    def __new__(cls, name, bases, attrs):
        jsoner_field = attrs.pop(_field_name, {})
        new_cls = super(MetaSerializable, cls).__new__(cls, name, bases, attrs)

        for base in bases:
            if hasattr(base, _field_name):
                jsoner_field.update(getattr(base, _field_name, {}))

        for k, v in attrs.items():
            if isinstance(v, BaseField):
                # Here's the place where the magic comes!
                jsoner_field[k] = v
                setattr(new_cls, k, None)

        setattr(new_cls, _field_name, jsoner_field)
        return new_cls

    def __call__(cls, *args, **kwargs):
        return type.__call__(cls, *args, **kwargs)


@with_metaclass(MetaSerializable)
class Serializable(object):

    def loads(self, data):
        # type: () -> self
        loaded = json.loads(data)
        if not isinstance(loaded, dict):
            raise AttributeError("data is not dict instance: '{}'".format(
                type(data).__name__
            ))
        for field_name, field in getattr(self, _field_name, {}).items():
            if isinstance(field, BaseField):
                j_field_name = pythonize_str(field_name) if field.jsonize else \
                               field_name
                j_value = loaded.get(j_field_name, None)
                loaded_value = field.loads(j_value)
                setattr(self, field_name, loaded_value)
        return self

    def dumps(self):
        # type: () -> self
        dumped = dict()
        for field_name, field in getattr(self, _field_name, {}).items():
            if isinstance(field, BaseField):
                value = getattr(self, field_name, None)
                j_field_name = jsonize_str(field_name) if field.jsonize else \
                               field_name
                j_value = field.dumps(value)
                dumped[j_field_name] = j_value
        return json.dumps(dumped)
