from __future__ import absolute_import
from __future__ import unicode_literals
import traceback
import logging


def init_logging(filename=None, level=None):
    logging_config = {"format": "%(asctime)-23s %(levelname)8s: %(message)s",
                      "level": level or logging.INFO}
    if filename is not None:
        logging_config["filename"] = filename
    logging.getLogger().handlers = []
    logging.basicConfig(**logging_config)


try:
    import threading
except ImportError:  # pragma: no cover
    threading = None


if threading:
    _lock = threading.RLock()
else:  # pragma: no cover
    _lock = None


def _acquire_lock():
    """
    Acquire the module-level lock for serializing access to shared data.

    This should be released with _release_lock().
    """
    if _lock:
        _lock.acquire()


def _release_lock():
    """
    Release the module-level lock acquired by calling _acquire_lock().
    """
    if _lock:
        _lock.release()


class Log(object):

    LEVEL = logging.DEBUG

    logger = logging.getLogger()
    logging_config = {"format": "%(asctime)-23s %(levelname)8s: %(message)s",
                      "level": logging.DEBUG}
    logging.basicConfig(**logging_config)

    @staticmethod
    def get_or_create_logger(tag):
        return logging.getLogger(tag)

    @staticmethod
    def log(level, tag, message, ex):
        # The tab should be the module.__name__ but not required!
        msg = "{}::{}".format(tag, message)
        if ex is not None:
            # exc_type, exc_value, exc_traceback = sys.exc_info()
            message = "%s\nTraceback:\n%s" % (msg, traceback.format_exc())
        logging.getLogger(tag).log(level, msg)

    @staticmethod
    def v(tag, message, ex=None):
        """ Thers is no VERBOSE in python logging we use DEBUG """
        return Log.log(logging.DEBUG, tag, message, ex)

    @staticmethod
    def d(tag, message, ex=None):
        return Log.log(logging.DEBUG, tag, message, ex)

    @staticmethod
    def i(tag, message, ex=None):
        return Log.log(logging.INFO, tag, message, ex)

    @staticmethod
    def w(tag, message, ex=None):
        return Log.log(logging.WARN, tag, message, ex)

    @staticmethod
    def e(tag, message, ex=None):
        return Log.log(logging.ERROR, tag, message, ex)
