from __future__ import unicode_literals
from __future__ import absolute_import


try:
    # OpenSSL's PKCS5_PBKDF2_HMAC requires OpenSSL 1.0+ with HMAC and SHA
    from _hashlib import pbkdf2_hmac
except ImportError:
    from hashlib import new
    import binascii
    import struct
    import sys

    try:
        range = xrange
    except NameError:
        unicode = str

    _trans_5C = bytearray(x ^ 0x5C for x in range(256))
    _trans_36 = bytearray(x ^ 0x36 for x in range(256))

    def data_to_bytes(data, encoding="utf-8"):
        if sys.version_info < (3, 0):
            return bytes(data)
        else:
            return bytes(data, encoding)

    def pbkdf2_hmac(hash_name, password, salt, iterations, dklen=None):
        """Password based key derivation function 2 (PKCS #5 v2.0)

        This Python implementations based on the hmac module about as fast
        as OpenSSL's PKCS5_PBKDF2_HMAC for short passwords and much faster
        for long passwords.
        """
        if not isinstance(hash_name, (str, unicode)):
            raise TypeError(hash_name)

        if not isinstance(password, (bytes, bytearray)):
            error = "a bytes-like object is required, not '{}'".format(
                password.__class__.__name__
            )
            raise TypeError(error)
        if not isinstance(salt, (bytes, bytearray)):
            error = "a bytes-like object is required, not '{}'".format(
                salt.__class__.__name__
            )
            raise TypeError(error)

        # Fast inline HMAC implementation
        inner = new(hash_name)
        outer = new(hash_name)
        blocksize = getattr(inner, 'block_size', 64)
        if len(password) > blocksize:
            password = new(hash_name, password).digest()
        password = password + b'\x00' * (blocksize - len(password))
        inner.update(password.translate(_trans_36))
        outer.update(password.translate(_trans_5C))

        def prf(msg, inner=inner, outer=outer):
            # PBKDF2_HMAC uses the password as key. We can re-use the same
            # digest objects and just update copies to skip initialization.
            icpy = inner.copy()
            ocpy = outer.copy()
            icpy.update(msg)
            ocpy.update(icpy.digest())
            return ocpy.digest()

        if iterations < 1:
            raise ValueError(iterations)
        if dklen is None:
            dklen = outer.digest_size
        if dklen < 1:
            raise ValueError(dklen)

        hex_format_string = "%%0%ix" % (new(hash_name).digest_size * 2)

        dkey = b''
        loop = 1
        while len(dkey) < dklen:
            prev = prf(salt + struct.pack(b'>I', loop))
            rkey = int(binascii.hexlify(prev), 16)
            for i in range(iterations - 1):
                prev = prf(prev)
                rkey ^= int(binascii.hexlify(prev), 16)
            loop += 1
            dkey += binascii.unhexlify(hex_format_string % rkey)

        return dkey[:dklen]


def pbkdf(secret, salt, hash="sha256", iterations=10000, dklen=128):
    return pbkdf2_hmac(hash, secret, salt, iterations, dklen)
