""" Kraken data packet implementation """
from dataclasses import dataclass
from typing import Dict

import marshmallow_dataclass
from marshmallow import post_dump, pre_load
from stringcase import camelcase, snakecase

from ..utils.functions import capitalize_first_char


@dataclass
class BasicDataclass:
    """ Kraken Packet """

    @pre_load
    def to_snake_case(self, data, **_kwargs):
        """ to snake case pre load method.
            Converts javaStyle parameters into the python_style syntax.
            :type data: dict
            :return str"""
        return {snakecase(key): value for key, value in data.items()}

    @post_dump
    def to_camel_case(self, data, **_kwargs):
        """ to camel case post load method """
        return {camelcase(key): value for key, value in data.items()}

    @classmethod
    def get_schema(cls, *args, **kwargs):
        """ Get schema """
        return marshmallow_dataclass.class_schema(cls)(*args, **kwargs)

    def dump(self, *args, **kwargs) -> Dict:
        """ Serialize to JSON object """
        return self.get_schema(*args, **kwargs).dump(self)

    @classmethod
    def load(cls, data, *args, **kwargs) -> "BasicDataclass":
        """ Deserialize from JSON object """
        return cls.get_schema(*args, **kwargs).load(data)
