""" marshmallow-dataclass need a mixin to work with camel - snake cases """
import uuid
from datetime import datetime
from time import time
from typing import Dict, Any, Union, List

import marshmallow_dataclass
from marshmallow import pre_load, post_dump, EXCLUDE
from stringcase import snakecase, camelcase


def uuid_factory() -> str:
    """ Set unique UUID """
    return uuid.uuid4().__str__()


def timestamp_factory() -> int:
    """ Set current unix timestamp """
    return int(time() * 1000)


class CamelCaseMixin:
    """ Camel Case mixin """

    class Meta:
        """ Meta class with configuration """
        unknown = EXCLUDE

    @pre_load
    def to_snake_case(self, data, **_kwargs):
        """ to snake case pre load method """
        return {snakecase(key): value for key, value in data.items()}

    @post_dump
    def to_camel_case(self, data, **_kwargs):
        """ to camel case post load method """
        return {camelcase(key): value for key, value in data.items()}

    @classmethod
    def get_schema(cls, *args, **kwargs):
        """ Get schema """
        return marshmallow_dataclass.class_schema(cls)(*args, **kwargs)

    @classmethod
    def load(cls, data: Dict[str, Any], *args, **kwargs) -> "CamelCaseMixin":
        """ load data into the object """
        schema = cls.get_schema(*args, **kwargs)
        return schema.load(data)

    def dump(self, *args, **kwargs) -> Union[Dict[str, Any], List]:
        """ Dump the object to json dict """
        return self.get_schema(*args, **kwargs).dump(self)
