from ...abstract_service import AbstractService
from ...utils.decorators import with_metaclass


class NamedParam(type):

    def __new__(mcs, name, bases, namespace):
        return type.__new__(mcs, name, bases, namespace)

    def __unicode__(cls):
        if getattr(cls, "name_brackets", False):
            return u"[{0}]".format(cls.__name__)
        return cls.__name__

    def __str__(cls):
        return cls.__unicode__()

    def __bytes__(cls):
        return cls.__unicode__().encode("utf-8")


@with_metaclass(NamedParam)
class ServiceParams(object):

    @with_metaclass(NamedParam)
    class Unit(object):

        name_brackets = True

        @with_metaclass(NamedParam)
        class Description(object):
            pass

        @with_metaclass(NamedParam)
        class After(object):
            pass

    @with_metaclass(NamedParam)
    class Service(object):

        name_brackets = True

        @with_metaclass(NamedParam)
        class Type(object):
            SIMPLE = "simple"
            FORKING = "forking"

        @with_metaclass(NamedParam)
        class Restart(object):
            ALWAYS = "always"

        @with_metaclass(NamedParam)
        class RestartSec(object):
            pass

        @with_metaclass(NamedParam)
        class User(object):
            pass

        @with_metaclass(NamedParam)
        class ExecStart(object):
            pass

        @with_metaclass(NamedParam)
        class Environment(object):
            pass

    @with_metaclass(NamedParam)
    class Install(object):

        name_brackets = True

        @with_metaclass(NamedParam)
        class WantedBy(object):
            pass


class SystemdService(AbstractService):
    name = None
    # [Unit]
    unit_description = None
    unit_after = None
    # [Service]
    service_type = None
    service_restart = None
    service_restart_sec = None
    service_user = None
    service_exec_start = None
    # [Install]
    install_wanted_by = None

    def set_name(self, name):
        """
        :param str name:
        :rtype: SystemdService
        """
        self.name = name
        return self

    def set_unit_description(self, unit_description):
        """
        :param str unit_description:
        :rtype: SystemdService
        """
        self.unit_description = unit_description
        return self

    def set_unit_after(self, unit_after):
        """
        :param str unit_after:
        :rtype: SystemdService
        """
        self.unit_after = unit_after
        return self

    def set_service_type(self, service_type):
        """
        :param str service_type:
        :rtype: SystemdService
        """
        self.service_type = service_type
        return self

    def set_service_restart(self, service_restart):
        """
        :param str service_restart:
        :rtype: SystemdService
        """
        self.service_restart = service_restart
        return self

    def set_service_restart_sec(self, service_restart_sec):
        """
        :param int service_restart_sec:
        :rtype: SystemdService
        """
        self.service_restart_sec = service_restart_sec
        return self

    def set_service_user(self, service_user):
        """
        :param str service_user:
        :rtype: SystemdService
        """
        self.service_user = service_user
        return self

    def set_service_exec_start(self, service_exec_start):
        """
        :param str service_exec_start:
        :rtype: SystemdService
        """
        self.service_exec_start = service_exec_start
        return self

    def set_install_wanted_by(self, install_wanted_by):
        """
        :param str install_wanted_by:
        :rtype: SystemdService
        """
        self.install_wanted_by = install_wanted_by
        return self

    @staticmethod
    def format_param(key, value=None, break_line=True):
        """
        :param Union[Type, str] key: Key name
        :param Union[str, int] value: The value
        :param bool break_line: If we break line at the end
        :rtype: str
        """
        serialized = "{key}".format(key=key)
        if value is not None:
            serialized = "{serialized}={value}".format(serialized=serialized,
                                                       value=value)
        if break_line:
            serialized += "\r\n"
        return serialized

    def to_string(self):
        # TODO(s1z): Add mandatory fields!
        # Serializing [Unit]
        unit_part = self.format_param(ServiceParams.Unit)
        if self.unit_description:
            unit_part += self.format_param(ServiceParams.Unit.Description,
                                           self.unit_description)
        if self.unit_after:
            unit_part += self.format_param(ServiceParams.Unit.After,
                                           self.unit_after)

        # Serializing [Service]
        service_part = self.format_param(ServiceParams.Service)
        if self.service_type:
            service_part += self.format_param(ServiceParams.Service.Type,
                                              self.service_type)

        if self.service_restart:
            service_part += self.format_param(ServiceParams.Service.Restart,
                                              self.service_restart)
        if self.service_restart_sec:
            service_part += self.format_param(ServiceParams.Service.RestartSec,
                                              self.service_restart_sec)
        if self.service_user:
            service_part += self.format_param(ServiceParams.Service.User,
                                              self.service_user)
        if self.service_exec_start:
            service_part += self.format_param(ServiceParams.Service.ExecStart,
                                              self.service_exec_start)
        # Serializing [Install]
        install_part = self.format_param(ServiceParams.Install)
        if self.install_wanted_by:
            install_part += self.format_param(ServiceParams.Install.WantedBy,
                                              self.install_wanted_by)

        return unit_part + service_part + install_part

    def __str__(self):
        return self.to_string()

    def __bytes__(self):
        return self.to_string().encode("utf-8")
