import asyncio
import logging
import os

import aiofiles
import aiofiles.os

from ..abstract_backend import AbstractBackend
from ...abstract_service import AbstractService


TAG = __name__
logger = logging.getLogger(TAG)


class SystemdBackend(AbstractBackend):

    SERVICES_DIR = "/etc/systemd/system"

    @classmethod
    async def is_supported(cls):
        """
        :rtype: bool
        """
        return await cls.execute_cmd("systemctl --version")

    @staticmethod
    async def execute_cmd(cmd_list):
        """
        :param List[str] cmd_list: List of commands
        :rtype: bool
        """
        proc = await asyncio.create_subprocess_shell(
            cmd_list,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE
        )
        await proc.communicate()
        if proc.returncode == 0:
            return True
        return False

    def daemon_reload(self):
        """
        :rtype: asyncio.Future[bool]
        """
        return self.execute_cmd("systemctl daemon-reload")

    async def create(self, service):
        """
        :param AbstractService service:
        :rtype: bool
        :return: returns True if service was created
        :exception: IOError
        """
        try:
            file_path = os.path.join(self.SERVICES_DIR, service.name)
            serialized_data = service.to_string()
            async with aiofiles.open(file_path, "w") as service_file:
                await service_file.write(serialized_data)
            return await self.daemon_reload()
        except IOError:
            logger.exception("{0}::create::error".format(TAG))
        return False

    async def delete(self, service):
        """
        :param AbstractService service:
        :rtype: bool
        :return: returns True if service was deleted
        :exception: IOError
        """
        try:
            file_path = os.path.join(self.SERVICES_DIR, service.name)
            if await aiofiles.os.stat(file_path):
                await aiofiles.os.remove(file_path)
            return self.daemon_reload()
        except (OSError, IOError):
            logger.exception("{0}::create::error".format(TAG))
        return False

    async def start(self, service):
        """
        :param AbstractService service:
        :rtype: bool
        :return: returns True if service was started
        :exception: IOError
        """
        cmd = "systemctl start {0}".format(service.name)
        return self.execute_cmd(cmd)

    async def stop(self, service):
        """
        :param AbstractService service:
        :rtype: bool
        :return: returns True if service was stopped
        :exception: IOError
        """
        cmd = "systemctl stop {0}".format(service.name)
        return self.execute_cmd(cmd)

    async def restart(self, service):
        """
        :param AbstractService service:
        :rtype: bool
        :return: returns True if service was restarted
        :exception: IOError
        """
        cmd = "systemctl restart {0}".format(service.name)
        return self.execute_cmd(cmd)
