package com.android.mobile.ui.fragments;

import android.app.ProgressDialog;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

import com.android.mobile.ApplicationContext;
import com.android.mobile.MainActivity;
import com.android.mobile.R;
import com.android.mobile.api.STBResponse;
import com.android.mobile.api.controller.AccountController;
import com.android.mobile.dependencies.Injector;
import com.android.mobile.util.AnimationUtil;
import com.android.mobile.util.ApplicationProvider;
import com.android.mobile.util.ApplicationProvider.Menu;
import com.android.mobile.util.FocusHandler;
import com.android.mobile.util.Preferences;

import javax.inject.Inject;

import butterknife.BindView;
import butterknife.ButterKnife;
import butterknife.OnClick;


public class MenuFragment extends Fragment implements AccountController.Callback {

    private static final String TAG = MenuFragment.class.getSimpleName();

    private ProgressDialog progressDialog;
    @Inject AccountController mAccountController;
    @Inject AnimationUtil animation;

    private MainActivity mActivity;

    @BindView(R.id.channels) public ImageView mChannelsView;
    @BindView(R.id.account) public ImageView mAccountView;
    @BindView(R.id.preferences) public ImageView mPreferenceView;
    @BindView(R.id.log_out) public ImageView mLogOutView;
    @BindView(R.id.title) public TextView mMenuTitle;

    private int current = 1;

    public static MenuFragment newInstance(Bundle bundle) {
        MenuFragment fragment = new MenuFragment();
        if (bundle != null) {
            fragment.setArguments(bundle);
        }
        return fragment;
    }

    public MenuFragment() {
        Injector.inject(this);
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        mActivity = (MainActivity) context;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mAccountController.setCallback(this);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.menu_fragment, container, false);
        ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        view.requestFocus();
        tryMove(Menu.CHANNELS);

        view.setOnKeyListener(new View.OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {

                boolean unique = event.getRepeatCount() == 0;
                boolean isDown = event.getAction() == KeyEvent.ACTION_DOWN;

                if (unique && isDown) {
                    switch (keyCode) {
                        case KeyEvent.KEYCODE_DPAD_LEFT:
                            tryMove(ApplicationProvider.MOVE_LEFT);
                            return true;

                        case KeyEvent.KEYCODE_DPAD_RIGHT:
                            tryMove(ApplicationProvider.MOVE_RIGHT);
                            return true;

                        case KeyEvent.KEYCODE_DPAD_CENTER:
                            performClick();
                            return true;
                    }
                }
                return false;
            }
        });
    }

    private void tryMove(int direction) {
        int focus = current + direction;

        if (focus >= 1 && focus <= 4) {

            ImageView view = getViewById(focus);
            animation.zoomIn(view);

            view = getViewById(current);
            animation.zoomOut(view);

            current = focus;
        }
        showMenuTitle();
    }

    private void showMenuTitle() {
        mMenuTitle.setText(getTitleById());
        animation.fadeOut(mMenuTitle);
        animation.fadeIn(mMenuTitle);
    }

    private boolean performClick() {
        switch (current) {
            case Menu.CHANNELS:
                onChannelsClick();
                return true;

            case Menu.ACCOUNT:
                onAccountClick();
                return true;

            case Menu.PREFERENCES:
                onPreferencesClick();
                return true;

            case Menu.LOGOUT:
                onLogOutClick();
                return true;
        }
        return false;
    }

    private String getTitleById() {
        switch (current) {
            case Menu.CHANNELS:
                return getString(R.string.menu__channels);

            case Menu.ACCOUNT:
                return getString(R.string.menu__private_office);

            case Menu.PREFERENCES:
                return getString(R.string.menu__preferences);

            case Menu.LOGOUT:
                return getString(R.string.menu__log_out);

            default:
                throw new AssertionError("Unknown identifier!");
        }
    }

    private ImageView getViewById(int id) {
        switch (id) {
            case Menu.CHANNELS:
                return mChannelsView;

            case Menu.ACCOUNT:
                return mAccountView;

            case Menu.PREFERENCES:
                return mPreferenceView;

            case Menu.LOGOUT:
                return mLogOutView;

            default:
                throw new AssertionError("Unknown identifier!");
        }
    }

    private FragmentTransaction getFragmentTransaction() {
        FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
        transaction.setCustomAnimations(R.anim.fade_scale_in, R.anim.fade_scale_out,
                R.anim.fade_scale_in, R.anim.fade_scale_out);
        return transaction;
    }

    @Override
    public void onResume() {
        super.onResume();

        FocusHandler.setMenuOpen(true);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        getFragmentManager().popBackStack(MenuFragment.class.getName(),
                FragmentManager.POP_BACK_STACK_INCLUSIVE);

        FocusHandler.setMenuOpen(false);
    }

    @Override
    public void onDetach() {
        super.onDetach();

        if (mActivity != null) {
            mActivity = null;
        }
    }

    @Override
    public void onRun() {
        progressDialog = ProgressDialog.show(getActivity(), getString(R.string.MenuFragment__logging_out),
                getString(R.string.MenuFragment__wait_please_this_can_take_a_while));
        progressDialog.setCancelable(false);
    }

    @Override
    public void onSuccess(STBResponse response) {
        logOut();

        progressDialog.dismiss();
        FocusHandler.setMenuOpen(false);

        getFragmentManager().popBackStack(MenuFragment.class.getName(),
                FragmentManager.POP_BACK_STACK_INCLUSIVE);

        mActivity.startActivity(new Intent(getActivity(), MainActivity.class));
        mActivity.overridePendingTransition(R.anim.fade_scale_in, R.anim.fade_scale_out);
    }

    @Override
    public void onFailure(Throwable throwable) {
        progressDialog.dismiss();

        Toast.makeText(getContext(), R.string.MenuFragment__sorry_something_went_wrong, Toast.LENGTH_SHORT)
                .show();
    }

    @OnClick(R.id.channels)
    public void onChannelsClick() {
        FocusHandler.setMenuOpen(false);

        getFragmentManager().popBackStack(MenuFragment.class.getName(),
                FragmentManager.POP_BACK_STACK_INCLUSIVE);

        getFragmentTransaction()
                .replace(R.id.container, ChannelsFragment.newInstance(null))
                .commit();
    }

    @OnClick(R.id.account)
    public void onAccountClick() {
        FocusHandler.setMenuOpen(false);

        getFragmentManager().popBackStack(MenuFragment.class.getName(),
                FragmentManager.POP_BACK_STACK_INCLUSIVE);

        getFragmentTransaction()
                .replace(R.id.container, AccountFragment.newInstance(null))
                .commit();
    }

    @OnClick(R.id.preferences)
    public void onPreferencesClick() {
        FocusHandler.setMenuOpen(false);

        getFragmentManager().popBackStack(MenuFragment.class.getName(),
                FragmentManager.POP_BACK_STACK_INCLUSIVE);

        getFragmentTransaction()
                .replace(R.id.container, PreferencesFragment.newInstance(null))
                .commit();
    }

    @OnClick(R.id.log_out)
    public void onLogOutClick() {
        mAccountController.signOut();
    }

    private void logOut() {
        Preferences.setUserLogin(getActivity(), null);
        Preferences.setUserPassword(getActivity(), null);
    }

}
