package com.android.mobile.dependencies.module;

import com.android.mobile.security.TrustStore;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.concurrent.TimeUnit;

import javax.inject.Named;
import javax.inject.Singleton;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import dagger.Module;
import dagger.Provides;
import okhttp3.OkHttpClient;


@Module
public class NetworkModule {

    private SSLSocketFactory sslSocketFactory;
    private X509TrustManager trustManager;

    public NetworkModule(TrustStore trustStore) {
        try {
            TrustManager[] trustManagers = getTrustManager(trustStore);
            SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, trustManagers, null);
            trustManager = (X509TrustManager) trustManagers[0];
            sslSocketFactory = context.getSocketFactory();
        } catch (NoSuchAlgorithmException | KeyManagementException e ){
            throw new AssertionError(e);
        }
    }

    @Provides
    @Singleton
    @Named("TLS")
    OkHttpClient provideSecureHttpClient() {
        return new OkHttpClient.Builder()
                .connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(10, TimeUnit.SECONDS)
                .sslSocketFactory(sslSocketFactory, trustManager)
                .hostnameVerifier(HttpsURLConnection.getDefaultHostnameVerifier())
                .build();
    }

    public TrustManager[] getTrustManager(TrustStore trustStore) {
        try {
            InputStream keyStoreInputStream = trustStore.getKeyStoreInputStream();
            KeyStore keyStore = KeyStore.getInstance("BKS");

            keyStore.load(keyStoreInputStream, trustStore.getKeyStorePassword().toCharArray());

            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance("X509");
            trustManagerFactory.init(keyStore);

            return trustManagerFactory.getTrustManagers();
        } catch (KeyStoreException | CertificateException | IOException | NoSuchAlgorithmException e) {
            throw new AssertionError(e);
        }
    }

}
