package com.android.mobile;

import com.android.mobile.database.model.ChannelRecord;
import com.android.mobile.dependencies.Injector;
import com.android.mobile.player.ExtractorRendererBuilder;
import com.android.mobile.player.HlsRendererBuilder;
import com.android.mobile.player.Player;
import com.android.mobile.ui.AutoHideView;
import com.android.mobile.ui.LoadingView;
import com.android.mobile.ui.divider.VerticalDividerItemDecoration;
import com.android.mobile.ui.adapters.LinearTrackSelectionAdapter;
import com.android.mobile.ui.snappysmoothscroller.SnapType;
import com.android.mobile.ui.snappysmoothscroller.SnappyLinearLayoutManager;
import com.android.mobile.util.AnimationUtil;
import com.android.mobile.util.ApplicationProvider;
import com.android.mobile.util.Dialogs;
import com.android.mobile.util.ViewUtil;
import com.google.android.exoplayer.ExoPlaybackException;
import com.google.android.exoplayer.MediaCodecTrackRenderer.DecoderInitializationException;
import com.google.android.exoplayer.MediaCodecUtil.DecoderQueryException;
import com.google.android.exoplayer.audio.AudioCapabilities;
import com.google.android.exoplayer.audio.AudioCapabilitiesReceiver;
import com.google.android.exoplayer.drm.UnsupportedDrmException;
import com.google.android.exoplayer.metadata.id3.ApicFrame;
import com.google.android.exoplayer.metadata.id3.GeobFrame;
import com.google.android.exoplayer.metadata.id3.Id3Frame;
import com.google.android.exoplayer.metadata.id3.PrivFrame;
import com.google.android.exoplayer.metadata.id3.TextInformationFrame;
import com.google.android.exoplayer.metadata.id3.TxxxFrame;
import com.google.android.exoplayer.util.Util;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.util.Log;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.animation.DecelerateInterpolator;
import android.widget.LinearLayout;
import android.widget.TextView;

import java.util.List;

import javax.inject.Inject;

import butterknife.BindView;
import butterknife.ButterKnife;


public class PlayerActivity extends Activity implements SurfaceHolder.Callback, Player.Listener,
        Player.Id3MetadataListener, AudioCapabilitiesReceiver.Listener {

  private static final String TAG = PlayerActivity.class.getSimpleName();

  public static final String CHANNEL_ID_EXTRA = "channel_id";

  private Player player;
  private boolean playerNeedsPrepare;
  private LinearTrackSelectionAdapter mChannelsAdapter;

  @BindView(R.id.surface_view) public SurfaceView surfaceView;
  @BindView(R.id.video_title) public TextView mVideoTitle;
  @BindView(R.id.player_header) public LinearLayout mPlayerHeader;

  private Uri contentUri;
  private int contentType;
  private ChannelRecord channelRecord;
  private SurfaceHolder mSurfaceHolder;

  @BindView(R.id.recycler_view) public RecyclerView mRecyclerView;
  @BindView(R.id.quick_channel_panel) public AutoHideView mQuickChannelsPanel;
  @BindView(R.id.progress_bar) public LoadingView mLoadingView;

  @Inject AnimationUtil animationUtil;
  private AudioCapabilitiesReceiver audioCapabilitiesReceiver;

  @Override
  public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    setContentView(R.layout.player_activity);
    ButterKnife.bind(this);
    Injector.inject(this);

    channelRecord = ApplicationProvider.getById(getIntent().getIntExtra(CHANNEL_ID_EXTRA, -1));

    initializeRecyclerView();

    mVideoTitle.setText(channelRecord.getTitle());

    mSurfaceHolder = surfaceView.getHolder();
    mSurfaceHolder.addCallback(this);

    audioCapabilitiesReceiver = new AudioCapabilitiesReceiver(this, this);
    audioCapabilitiesReceiver.register();
  }

  private void highlightCurrentChannel() {
    int playing = ApplicationProvider.getCurrent();
    int focused = mChannelsAdapter.getFocusedItem();
    if (playing != focused)  {
      mChannelsAdapter.setFocused(playing);
      mChannelsAdapter.tryMoveSelection(mRecyclerView, ApplicationProvider.HOLD);
    }
  }

  private void initializeRecyclerView() {
    mQuickChannelsPanel.requestFocus();

    mRecyclerView.addItemDecoration(new VerticalDividerItemDecoration.Builder(this)
            .sizeResId(R.dimen.divider_vertical_margin)
            .colorResId(android.R.color.transparent)
            .build());

    mRecyclerView.setHasFixedSize(true);
    mRecyclerView.setItemViewCacheSize(32);
    mRecyclerView.setDrawingCacheEnabled(true);
    mRecyclerView.setDrawingCacheQuality(View.DRAWING_CACHE_QUALITY_HIGH);
    mRecyclerView.setLayerType(View.LAYER_TYPE_HARDWARE, null);
    mRecyclerView.getItemAnimator().setChangeDuration(5);
    mRecyclerView.getItemAnimator().setMoveDuration(100);

    mChannelsAdapter = new LinearTrackSelectionAdapter(this, ApplicationProvider.getChannels());
    mRecyclerView.setAdapter(mChannelsAdapter);

    SnappyLinearLayoutManager snappyLinearLayoutManager = new SnappyLinearLayoutManager(this, LinearLayoutManager.HORIZONTAL, false);
    snappyLinearLayoutManager.setSnapType(SnapType.CENTER);
    snappyLinearLayoutManager.setSnapInterpolator(new DecelerateInterpolator());

    mRecyclerView.setLayoutManager(snappyLinearLayoutManager);
    mRecyclerView.setOnKeyListener(new View.OnKeyListener() {
      @Override
      public boolean onKey(View v, int keyCode, KeyEvent event) {
        if (event.getAction() == KeyEvent.ACTION_DOWN) {
          boolean isUnique = event.getRepeatCount() == 0;
          if (isUnique) {
            mQuickChannelsPanel.reset();
            switch (keyCode) {
              case KeyEvent.KEYCODE_DPAD_RIGHT:
                return mChannelsAdapter.tryMoveSelection(mRecyclerView, ApplicationProvider.MOVE_RIGHT);

              case KeyEvent.KEYCODE_DPAD_LEFT:
                return mChannelsAdapter.tryMoveSelection(mRecyclerView, ApplicationProvider.MOVE_LEFT);

              case KeyEvent.KEYCODE_DPAD_CENTER:
                startNewChannel();
                return true;
            }
          }
        }
        return false;
      }
    });

    highlightCurrentChannel();
  }

  private void startNewChannel() {

    int mFocusChannel = mChannelsAdapter.getFocusedItem();
    int mCurrentChannel = ApplicationProvider.getCurrent();

    if (mFocusChannel == mCurrentChannel) {
      hidePlayerPanel();
    } else {
      ChannelRecord focusedChannel = ApplicationProvider.getById(mFocusChannel);
      mVideoTitle.setText(focusedChannel.getTitle());
      releasePlayer();
      contentUri = Uri.parse(focusedChannel.getUrl());
      preparePlayer(true);
      ApplicationProvider.setCurrent(mFocusChannel);
    }

  }

  @Override
  public boolean onKeyDown(int keyCode, KeyEvent event) {
    final boolean unique = event.getRepeatCount() == 0;
    if (unique) {
      if (keyCode == KeyEvent.KEYCODE_BACK) {
        if (ViewUtil.isVisible(mQuickChannelsPanel)) {
          hidePlayerPanel();
        } else {
          super.onBackPressed();
        }
        return true;
      } else if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER) {
        if (ViewUtil.isVisible(mQuickChannelsPanel)) {
          startNewChannel();
        } else {
          showPlayerPanel();
        }
        return true;
      }
    }
    return false;
  }

  private void hidePlayerPanel() {
    mQuickChannelsPanel.hide();
    mRecyclerView.clearFocus();
  }

  private void showPlayerPanel() {
    mQuickChannelsPanel.show();
    highlightCurrentChannel();
    mRecyclerView.requestFocus();
  }

  @Override
  public boolean onTouchEvent(MotionEvent event) {
    showPlayerPanel();
    return false;
  }

  @Override
  public void onNewIntent(Intent intent) {
    releasePlayer();
    setIntent(intent);
  }

  @Override
  public void onStart() {
    super.onStart();
    if (Util.SDK_INT > Build.VERSION_CODES.M) {
      onShown();
    }
  }

  @Override
  public void onResume() {
    super.onResume();
    if (Util.SDK_INT <= Build.VERSION_CODES.M || player == null) {
      onShown();
    }
  }

  private void onShown() {
    contentUri = Uri.parse(channelRecord.getUrl());
    contentType = Util.inferContentType(contentUri.getLastPathSegment());

    if (player == null) {
        preparePlayer(true);
    } else {
      player.setBackgrounded(false);
    }
  }

  @Override
  public void onPause() {
    super.onPause();
    if (Util.SDK_INT <= Build.VERSION_CODES.M) {
      onHidden();
    }
  }

  @Override
  public void onStop() {
    super.onStop();
    if (Util.SDK_INT > Build.VERSION_CODES.M) {
      onHidden();
    }
  }

  private void onHidden() {
    releasePlayer();
  }

  @Override
  public void onDestroy() {
    super.onDestroy();
    audioCapabilitiesReceiver.unregister();
    releasePlayer();
  }

  @Override
  public void onAudioCapabilitiesChanged(AudioCapabilities audioCapabilities) {
    if (player == null) {
      return;
    }
    boolean backgrounded = player.getBackgrounded();
    boolean playWhenReady = player.getPlayWhenReady();
    releasePlayer();
    preparePlayer(playWhenReady);
    player.setBackgrounded(backgrounded);
  }

  private Player.RendererBuilder getRendererBuilder() {
    String userAgent = Util.getUserAgent(this, BuildConfig.USER_AGENT);
    switch (contentType) {
      case Util.TYPE_HLS:
        return new HlsRendererBuilder(this, userAgent, contentUri.toString());
      case Util.TYPE_OTHER:
        return new ExtractorRendererBuilder(this, userAgent, contentUri);
      default:
        throw new IllegalStateException("Unsupported type: " + contentType);
    }
  }

  private void preparePlayer(boolean playWhenReady) {
    if (player == null) {
      player = new Player(getRendererBuilder());
      player.addListener(this);
      player.setMetadataListener(this);
      playerNeedsPrepare = true;
    }
    if (playerNeedsPrepare) {
      player.prepare();
      playerNeedsPrepare = false;
    }
    player.setSurface(mSurfaceHolder.getSurface());
    player.setPlayWhenReady(playWhenReady);
  }

  private void releasePlayer() {
    if (player != null) {
      player.release();
      player = null;
    }
  }

  private void showBuffering() {
    mLoadingView.setVisibility(true);
  }

  private void hideBuffering() {
    mLoadingView.setVisibility(false);
  }

  @Override
  public void onStateChanged(boolean playWhenReady, int playbackState) {
    switch (playbackState) {
      case Player.STATE_IDLE:
        break;
      case Player.STATE_PREPARING:
      case Player.STATE_BUFFERING:
        showBuffering();
        break;
      case Player.STATE_READY:
        hideBuffering();
        break;
      case Player.STATE_ENDED:
        break;
      case Player.TRACK_DISABLED:
        break;
      case Player.TRACK_DEFAULT:
        break;
    }
  }

  @Override
  public void onError(Exception e) {
    String errorString = null;
    if (e instanceof UnsupportedDrmException) {
      // Special case DRM failures.
      UnsupportedDrmException unsupportedDrmException = (UnsupportedDrmException) e;
      errorString = getString(Util.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2 ? R.string.player_activity__error_drm_not_supported
          : unsupportedDrmException.reason == UnsupportedDrmException.REASON_UNSUPPORTED_SCHEME
          ? R.string.player_activity__error_drm_unsupported_scheme : R.string.player_activity__error_drm_unknown);
    } else if (e instanceof ExoPlaybackException
        && e.getCause() instanceof DecoderInitializationException) {
      // Special case for decoder initialization failures.
      DecoderInitializationException decoderInitializationException =
          (DecoderInitializationException) e.getCause();
      if (decoderInitializationException.decoderName == null) {
        if (decoderInitializationException.getCause() instanceof DecoderQueryException) {
          errorString = getString(R.string.player_activity__error_querying_decoders);
        } else if (decoderInitializationException.secureDecoderRequired) {
          errorString = getString(R.string.player_activity__error_no_secure_decoder,
              decoderInitializationException.mimeType);
        } else {
          errorString = getString(R.string.player_activity__error_no_decoder,
              decoderInitializationException.mimeType);
        }
      } else {
        errorString = getString(R.string.player_activity__error_instantiating_decoder,
            decoderInitializationException.decoderName);
      }
    } else if (e instanceof ExoPlaybackException) {
      errorString = getString(R.string.player_activity__video_format_not_allowed);
    }
    if (errorString != null) {
      Dialogs.showDialog(this, getString(R.string.player_activity__something_went_wrong), errorString);
    }
    playerNeedsPrepare = true;
  }

  @Override
  public void onVideoSizeChanged(int width, int height, int unappliedRotationDegrees, float pixelWidthAspectRatio) {

  }

  @Override
  public void onId3Metadata(List<Id3Frame> id3Frames) {
    for (Id3Frame id3Frame : id3Frames) {
      if (id3Frame instanceof TxxxFrame) {
        TxxxFrame txxxFrame = (TxxxFrame) id3Frame;
        Log.i(TAG, String.format("ID3 TimedMetadata %s: description=%s, value=%s", txxxFrame.id,
            txxxFrame.description, txxxFrame.value));
      } else if (id3Frame instanceof PrivFrame) {
        PrivFrame privFrame = (PrivFrame) id3Frame;
        Log.i(TAG, String.format("ID3 TimedMetadata %s: owner=%s", privFrame.id, privFrame.owner));
      } else if (id3Frame instanceof GeobFrame) {
        GeobFrame geobFrame = (GeobFrame) id3Frame;
        Log.i(TAG, String.format("ID3 TimedMetadata %s: mimeType=%s, filename=%s, description=%s",
            geobFrame.id, geobFrame.mimeType, geobFrame.filename, geobFrame.description));
      } else if (id3Frame instanceof ApicFrame) {
        ApicFrame apicFrame = (ApicFrame) id3Frame;
        Log.i(TAG, String.format("ID3 TimedMetadata %s: mimeType=%s, description=%s",
                apicFrame.id, apicFrame.mimeType, apicFrame.description));
      } else if (id3Frame instanceof TextInformationFrame) {
        TextInformationFrame textInformationFrame = (TextInformationFrame) id3Frame;
        Log.i(TAG, String.format("ID3 TimedMetadata %s: description=%s", textInformationFrame.id,
            textInformationFrame.description));
      } else {
        Log.i(TAG, String.format("ID3 TimedMetadata %s", id3Frame.id));
      }
    }
  }

  @Override
  public void surfaceCreated(SurfaceHolder holder) {
    mSurfaceHolder = holder;
    if (player != null) {
      player.setSurface(mSurfaceHolder.getSurface());
    }
  }

  @Override
  public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
  }

  @Override
  public void surfaceDestroyed(SurfaceHolder holder) {
    if (player != null) {
      player.blockingClearSurface();
    }
  }

}
