package com.android.mobile.ui.fragments;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;

import com.android.mobile.ApplicationContext;
import com.android.mobile.MainActivity;
import com.android.mobile.R;
import com.android.mobile.api.STBResponse;
import com.android.mobile.api.controller.AccountController;
import com.android.mobile.api.retrofit.service.AccountService;
import com.android.mobile.dependencies.Injector;
import com.android.mobile.ui.LoadingView;
import com.android.mobile.util.AnimationUtil;
import com.android.mobile.util.FocusHandler;
import com.android.mobile.util.Preferences;
import com.android.mobile.util.Dialogs;
import com.android.mobile.util.Util;

import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;

import butterknife.BindView;
import butterknife.ButterKnife;
import butterknife.OnClick;


public class SignInFragment extends Fragment implements AccountController.Callback {

    private static final String TAG = SignInFragment.class.getSimpleName();

    public static final int ID = 3;

    @BindView(R.id.sign_in) public Button mSignIn;
    @BindView(R.id.email) public EditText mEmail;
    @BindView(R.id.password) public EditText mPassword;
    @BindView(R.id.content) public LinearLayout mContent;
    @BindView(R.id.loading_view) public LoadingView mLoadingView;

    @Inject AnimationUtil mAnimationUtil;
    @Inject AccountController mAccountController;

    public static SignInFragment newInstance(Bundle bundle) {
        SignInFragment instance = new SignInFragment();
        if (bundle != null) {
            instance.setArguments(bundle);
        }
        return instance;
    }

    public SignInFragment() {
        Injector.inject(this);
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mAccountController.setCallback(this);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.sign_in_fragment, container, false);
        ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        FocusHandler.setVisible(ID);
        mEmail.requestFocus();
    }

    @Override
    public void onRun() {
        setLoading(true);
    }

    @Override
    public void onSuccess(STBResponse response) {
        setLoading(false);
        Log.d(TAG, "onSuccess: " + response.toString());

        if (response.isSuccessful()) {
            Activity activity = getActivity();

            Preferences.setUserLogin(getContext(), mEmail.getText().toString());
            Preferences.setUserPassword(getContext(), mPassword.getText().toString());

            startActivity(new Intent(activity, MainActivity.class));
            activity.finish();
        }
    }

    @Override
    public void onFailure(Throwable throwable) {
        setLoading(false);

        Log.d(TAG, "onFailure: " + throwable.getMessage());

        Dialogs.showDialog(getContext(),
                getString(R.string.sign_in_fragment__something_went_wrong),
                throwable.getMessage());

        mEmail.requestFocus();
    }

    @OnClick(R.id.sign_in) public void signIn() {
        String email = mEmail.getText().toString();
        String password = mPassword.getText().toString();

        if (email.isEmpty()) {
            mEmail.setError(getString(R.string.sign_in_fragment__please_fill_this_field));
            mAnimationUtil.wobble(mEmail);
            return;
        }

        if (!Util.isValidEmail(email)) {
            mEmail.setError(getString(R.string.sign_in_fragment__please_enter_an_email));
            mAnimationUtil.wobble(mEmail);
            return;
        }

        if (password.isEmpty()) {
            mPassword.setError(getString(R.string.sign_in_fragment__please_fill_this_field));
            mAnimationUtil.wobble(mPassword);
            return;
        }

        if (!Util.isValidPassword(password)) {
            mPassword.setError(getString(R.string.sign_in_fragment__please_enter_a_password));
            mAnimationUtil.wobble(mPassword);
            return;
        }

        Map<String, String> params = new HashMap<>();
            params.put(AccountService.EMAIL, email);
            params.put(AccountService.PASSWORD, Util.md5(password));
            params.put(AccountService.SERIAL, Preferences.getUserIdentifier(getContext()));

        mAccountController.signIn(params);
    }

    private void setLoading(boolean isLoading) {
        mLoadingView.setVisibility(isLoading);
        mContent.setVisibility(isLoading ? View.GONE : View.VISIBLE);
    }

}
