package com.android.mobile.ui.fragments;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.android.mobile.ApplicationContext;
import com.android.mobile.MainActivity;
import com.android.mobile.R;
import com.android.mobile.api.STBResponse;
import com.android.mobile.api.controller.AccountController;
import com.android.mobile.api.retrofit.service.AccountService;
import com.android.mobile.dependencies.Injector;
import com.android.mobile.ui.LoadingView;
import com.android.mobile.util.APIUtil;
import com.android.mobile.util.AnimationUtil;
import com.android.mobile.util.Preferences;
import com.android.mobile.util.Dialogs;
import com.android.mobile.util.Util;
import com.android.mobile.util.ViewUtil;

import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;

import butterknife.BindView;
import butterknife.ButterKnife;
import butterknife.OnClick;


public class CheckSerialFragment extends Fragment implements AccountController.Callback {

    private static final String TAG = CheckSerialFragment.class.getSimpleName();

    public static final int ID = 1;

    @BindView(R.id.loading_view) public LoadingView mLoadingView;
    @BindView(R.id.check_serial_button) public Button mCheckSerial;
    @BindView(R.id.change_serial_button) public Button mChangeSerial;
    @BindView(R.id.serial) public TextView mSerial;
    @BindView(R.id.content) public LinearLayout mContent;

    @Inject AccountController mAccountController;
    @Inject AnimationUtil animationUtil;

    public static CheckSerialFragment newInstance(Bundle bundle) {
        CheckSerialFragment instance = new CheckSerialFragment();
        if (bundle != null) {
            instance.setArguments(bundle);
        }
        return instance;
    }

    public CheckSerialFragment() {
        Injector.inject(this);
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mAccountController.setCallback(this);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.check_serial_fragment, container, false);
        ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        mSerial.setText(Preferences.getUserIdentifier(getContext()));
    }

    @OnClick(R.id.check_serial_button)
    public void checkSerial() {

        String serial = mSerial.getText().toString();

        if (serial.isEmpty()) {
            mSerial.setError(getString(R.string.check_serial_fragment__save_serial));
            animationUtil.wobble(mSerial);
            return;
        }

        if (!Util.isValidSerial(serial)) {
            mSerial.setError(getString(R.string.check_serial_fragment__serial_is_not_valid));
            animationUtil.wobble(mSerial);
            return;
        }

        Map<String, String> params = new HashMap<>();
            params.put(AccountService.SERIAL, serial);

        mAccountController.setUserIdentifier(params);
    }

    @OnClick(R.id.change_serial_button)
    public void changeSerial() {
        getFragmentManager().beginTransaction()
                .replace(R.id.container, ChangeSerialFragment.newInstance(null))
                .addToBackStack(ChangeSerialFragment.class.getSimpleName())
                .commit();
    }

    @Override
    public void onRun() {
        setLoading(true);
    }

    @Override
    public void onSuccess(STBResponse response) {
        setLoading(false);
        Log.d(TAG, "onSuccess: " + response.toString());

        Map<String, String> data = response.getData();
        String isp = data.get("isp");

        ViewUtil.setImageDrawable(getActivity(),
                (ImageView) getActivity().findViewById(R.id.logotype),
                APIUtil.getISPLogo(isp != null ? isp : "0"));

        Preferences.setUserIsp(getContext(), isp);

        Preferences.setUserIdentifier(getContext(), mSerial.getText().toString());

        Activity activity = getActivity();
        startActivity(new Intent(activity, MainActivity.class));
        activity.finish();

    }

    @Override
    public void onFailure(Throwable throwable) {
        setLoading(false);

        Log.d(TAG, "onFailure: " + throwable.getMessage());

        Dialogs.showDialog(getContext(),
                getString(R.string.check_serial_fragment__something_went_wrong),
                throwable.getMessage());

        mCheckSerial.requestFocus();
    }

    private void setLoading(boolean isLoading) {
        mLoadingView.setVisibility(isLoading);
        mContent.setVisibility(isLoading ? View.GONE : View.VISIBLE);
    }

}
